/* $Id: tachreset.c,v 1.6 1998/09/16 02:05:22 ericb Exp $ */
/* Copyright (C) 1998, Hewlett-Packard Company, all rights reserved. */
/* Written by Eric Backus */

/* There is a potential problem with the tach board, where resetting
   it after it is set up for sync may not break the tach board out of
   the waiting-for-sync stuff.  This program tests that this problem
   is fixed. */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <unistd.h>
#include "e1432.h"
#include "err1432.h"

#define	NMOD_MAX	4

/* Wrap this around all the many function calls which might fail */
#ifdef	__lint
#define	CHECK(func)	\
do {\
    int _s = (func);\
    if (_s < 0)\
    {\
	(void) fprintf(stderr, "%s: %s: returned %d\n", progname, #func, _s);\
	return _s;\
    }\
} while (func)
#else
#define	CHECK(func)	\
do {\
    int _s = (func);\
    if (_s < 0)\
    {\
	(void) fprintf(stderr, "%s: %s: returned %d\n", progname, #func, _s);\
	return _s;\
    }\
} while (0)
#endif

static const volatile char rcsid[] =
"@(#)$Id: tachreset.c,v 1.6 1998/09/16 02:05:22 ericb Exp $";
static const char *progname;

static int
init(int nmod, SHORTSIZ16 *laddr, E1432ID *hw)
{
    struct e1432_hwconfig hwconfig[NMOD_MAX];
    int     i, tachs;

    /* Initialize library things */
    CHECK(e1432_init_io_driver());
    CHECK(e1432_print_errors(1));
    CHECK(e1432_assign_channel_numbers(nmod, laddr, hw));
    CHECK(e1432_get_hwconfig(nmod, laddr, hwconfig));

    tachs = 0;
    for (i = 0; i < nmod; i++)
	tachs += hwconfig[i].tach_chans;

    if (tachs == 0)
    {
	(void) fprintf(stderr, "%s: requires a tach channel\n", progname);
	return -1;
    }

    return 0;
}

static int
monitor_rpm(E1432ID hw, int chan,
	    int expect_zero, int expect_identical, int verbose)
{
    FLOATSIZ32 rpm, rpm_save;
    int     i, first, got_different;

    first = 1;
    got_different = 0;
    for (i = 0; i < 8; i++)
    {
	CHECK(e1432_get_current_rpm(hw, chan, &rpm));
	if (verbose)
	    (void) printf("rpm %g\n", rpm);

	if (first)
	{
	    rpm_save = rpm;
	    first = 0;
	}
	if (rpm_save != rpm)
	    got_different = 1;

	if (expect_zero && rpm != 0)
	{
	    (void) fprintf(stderr, "%s: expect RPM 0, got %g\n",
			   progname, rpm);
	    return -1;
	}
	if (expect_identical && got_different)
	{
	    (void) fprintf(stderr, "%s: expect same RPM, got %g, %g\n",
			   progname, rpm, rpm_save);
	    return -1;
	}

	(void) sleep(1);
    }

    /* We're relying on noise to make the RPM values vary by a little,
       unless the tach reset bug still exists which will make all RPM
       values be the same. */
    if (!expect_identical && !got_different)
    {
	(void) fprintf(stderr, "%s: RPM frozen at %g\n",
		       progname, rpm_save);
	return -1;
    }

    return 0;
}

static int
run(E1432ID hw, int chan, int verbose)
{
    if (verbose)
	(void) printf("Using tach channel %d\n", chan);

    chan = E1432_TACH_CHAN(chan);

    if (verbose)
	(void) printf("Tach inactive\n");

    /* RPM values should be zero from inactive tach channels */
    CHECK(monitor_rpm(hw, chan, 1, 1, verbose));

    if (verbose)
	(void) printf("Tach active\n");
    CHECK(e1432_set_active(hw, chan, E1432_CHANNEL_ON));
    CHECK(e1432_set_trigger_level(hw, chan, E1432_TRIGGER_LEVEL_LOWER, 0.8));
    CHECK(e1432_set_trigger_level(hw, chan, E1432_TRIGGER_LEVEL_UPPER, 1.4));

    /* RPM values should vary slightly due to noise */
    CHECK(monitor_rpm(hw, chan, 0, 0, verbose));

    if (verbose)
	(void) printf("After e1432_init_measure_to_booted\n");
    CHECK(e1432_init_measure_to_booted(hw, chan, 1));

    /* RPM values should be frozen while waiting for sync */
    CHECK(monitor_rpm(hw, chan, 0, 1, verbose));

    if (verbose)
	(void) printf("After e1432_reset_measure\n");
    CHECK(e1432_reset_measure(hw, chan));

    /* RPM values should vary slightly after reset */
    CHECK(monitor_rpm(hw, chan, 0, 0, verbose));

    if (verbose)
	(void) printf("After e1432_init_measure\n");
    CHECK(e1432_init_measure(hw, chan));

    /* RPM values should vary slightly after init */
    CHECK(monitor_rpm(hw, chan, 0, 0, verbose));

    if (verbose)
	(void) printf("Tach inactive\n");
    CHECK(e1432_set_active(hw, chan, E1432_CHANNEL_OFF));

    /* RPM values should be zero from inactive tach channels */
    CHECK(monitor_rpm(hw, chan, 1, 1, verbose));

    return 0;
}

static void
usage(void)
{
    (void) fprintf(stderr,
		   "Usage: %s [-2uvV] [-L laddr] [-N nmod]\n"
		   "\t-2: Use tach channel 2 instead of 1\n"
		   "\t-L: Logical address is <laddr>, default 8\n"
		   "\t-N: Use <nmod> modules, default 1\n"
		   "\t-u: Print this usage message\n"
		   "\t-v: Verbose output\n"
		   "\t-V: Print version info\n",
		   progname);
    exit(2);
}

int
main(int argc, char **argv)
{
    E1432ID hw;
    SHORTSIZ16 laddr[NMOD_MAX];
    char   *p;
    int     c, i, chan, nmod, verbose;

    /* Get program name */
    progname = strrchr(argv[0], '/');
    if (progname == NULL)
	progname = argv[0];
    else
	progname++;

    /* Set option defaults */
    chan = 1;
    laddr[0] = 8;
    nmod = 1;
    verbose = 0;

    /* Process command-line options */
    while ((c = getopt(argc, argv, "2L:N:uvV")) != -1)
	switch (c)
	{
	case '2':
	    chan = 2;
	    break;
	case 'L':
	    laddr[0] = (SHORTSIZ16) strtol(optarg, &p, 0);
	    if (optarg == p || laddr[0] < 0 || laddr[0] > 255)
	    {
		(void) fprintf(stderr,
			       "%s: invalid logical address: '%s'\n",
			       progname, optarg);
		usage();
	    }
	    break;
	case 'N':
	    nmod = strtol(optarg, &p, 0);
	    if (optarg == p || nmod < 0 || nmod > NMOD_MAX)
	    {
		(void) fprintf(stderr,
			       "%s: invalid number of modules: '%s'\n",
			       progname, optarg);
		usage();
	    }
	    break;
	case 'v':
	    verbose++;
	    break;
	case 'V':
	    (void) printf("%s\n", rcsid);
	    exit(EXIT_SUCCESS);
	case 'u':
	default:
	    usage();
	}

    if (argc > optind)
    {
	(void) fprintf(stderr, "%s: extra command-line arguments\n",
		       progname);
	usage();
    }

    /* Assume logical addresses are consecutive */
    for (i = 1; i < nmod; i++)
	laddr[i] = laddr[i - 1] + 1;

    /* Run the measurement */
    if (init(nmod, laddr, &hw) < 0)
	return EXIT_FAILURE;
    if (run(hw, chan, verbose) < 0)
	return EXIT_FAILURE;

    return EXIT_SUCCESS;
}
